import 'package:flutter/material.dart';
import 'dart:convert';
import 'package:http/http.dart' as http;
import 'package:slpl_app/config.dart';

class MyLeagueTableScreen extends StatefulWidget {
  @override
  _MyLeagueTableScreenState createState() => _MyLeagueTableScreenState();
}

class _MyLeagueTableScreenState extends State<MyLeagueTableScreen> {
  late Future<List<Team>> futureTeams;

  @override
  void initState() {
    super.initState();
    futureTeams = fetchTeams();
  }

  Future<List<Team>> fetchTeams() async {
    final response =
        await http.get(Uri.parse('${AppConfig.baseUrl}male_table.php'));
    if (response.statusCode == 200) {
      final Map<String, dynamic> data = json.decode(response.body);
      // Assuming your data is structured like {"team1": {...}, "team2": {...}, ...}
      List<Team> teams = data.entries
          .map((entry) => Team.fromJson({...entry.value, 'name': entry.key}))
          .toList();
      return teams;
    } else {
      throw Exception('Failed to load teams');
    }
  }

  @override
  Widget build(BuildContext context) {
    return FutureBuilder<List<Team>>(
      future: futureTeams,
      builder: (context, snapshot) {
        if (snapshot.hasData) {
          return LeagueTableScreen(teams: snapshot.data!);
        } else if (snapshot.hasError) {
          print(snapshot.error);
          return Center(child: Text('Error: ${snapshot.error}'));
        }
        return Center(child: CircularProgressIndicator());
      },
    );
  }
}

class MyMenTable extends StatelessWidget {
  const MyMenTable({Key? key}) : super(key: key);

  @override
  Widget build(BuildContext context) {
    return Scaffold(
      body: MyLeagueTableScreen(),
    );
  }
}

class LeagueTableScreen extends StatelessWidget {
  final List<Team> teams;

  const LeagueTableScreen({Key? key, required this.teams}) : super(key: key);

  Widget _buildDataTableCell(String text, {Color? textColor}) {
    return Container(
      color: Colors.white,
      padding: EdgeInsets.symmetric(horizontal: 6.0, vertical: 8.0),
      child: Text(
        text,
        style: TextStyle(
          color: textColor ?? const Color.fromARGB(255, 0, 0, 0),
          fontSize: 16,
          fontWeight: FontWeight.bold,
        ),
      ),
    );
  }

  @override
  Widget build(BuildContext context) {
    return Scaffold(
      appBar: AppBar(
        backgroundColor: Color.fromARGB(255, 16, 99, 60),
        foregroundColor: const Color.fromARGB(255, 252, 252, 252),
        leading: IconButton(
          icon: Icon(Icons.arrow_back),
          onPressed: () {
            Navigator.pop(context);
          },
        ),
        title: Container(
          decoration: BoxDecoration(
            borderRadius: BorderRadius.circular(8),
          ),
          child: const Text(
            "Table",
            style: TextStyle(
              color: Color.fromARGB(255, 246, 247, 248),
              fontSize: 20,
              fontWeight: FontWeight.bold,
            ),
          ),
        ),
      ),
      body: SingleChildScrollView(
        scrollDirection: Axis.horizontal,
        child: DataTable(
          columnSpacing: 0.0,
          columns: [
            DataColumn(label: _buildDataTableCell('Pos')),
            DataColumn(label: _buildDataTableCell('Team')),
            DataColumn(label: _buildDataTableCell('Played')),
            DataColumn(label: _buildDataTableCell('Won')),
            DataColumn(label: _buildDataTableCell('Drawn')),
            DataColumn(label: _buildDataTableCell('Lost')),
            DataColumn(label: _buildDataTableCell('GF')),
            DataColumn(label: _buildDataTableCell('GA')),
            DataColumn(label: _buildDataTableCell('GD')),
            DataColumn(label: _buildDataTableCell('Pts')),
          ],
          rows: teams.map((team) {
            return DataRow(
              cells: [
                DataCell(_buildDataTableCell(team.position?.toString() ?? '-')),
                DataCell(
                  Container(
                    alignment: Alignment.centerLeft,
                    child: Row(
                      children: [
                        SizedBox(width: 10),
                        Image.network(
                          team.imageUrl,
                          width: 40,
                          height: 40,
                          errorBuilder: (context, error, stackTrace) {
                            return const Placeholder(
                              fallbackWidth: 40,
                              fallbackHeight: 40,
                              color: Colors.grey, // Placeholder color
                            );
                          },
                        ),
                        SizedBox(width: 15),
                        Text(team.name),
                        SizedBox(width: 40),
                      ],
                    ),
                  ),
                ),
                DataCell(_buildDataTableCell(team.played.toString())),
                DataCell(_buildDataTableCell(team.won.toString())),
                DataCell(_buildDataTableCell(team.draw.toString())),
                DataCell(_buildDataTableCell(team.lost.toString())),
                DataCell(_buildDataTableCell(team.goalsScored.toString())),
                DataCell(_buildDataTableCell(team.goalsAgainst.toString())),
                DataCell(_buildDataTableCell(team.goalDifference.toString())),
                DataCell(_buildDataTableCell(team.points.toString())),
              ],
            );
          }).toList(),
        ),
      ),
    );
  }
}

class Team {
  final String position;
  final String name;
  final String played;
  final String won;
  final String draw;
  final String lost;
  final String goalsScored;
  final String goalsAgainst;
  final String goalDifference;
  final String points;
  final String imageUrl;

  Team({
    required this.position,
    required this.name,
    required this.played,
    required this.won,
    required this.draw,
    required this.lost,
    required this.goalsScored,
    required this.goalsAgainst,
    required this.goalDifference,
    required this.points,
    required this.imageUrl,
  });

  factory Team.fromJson(Map<String, dynamic> json) => Team(
        position: json['position'].toString(),
        name: json['name'].toString(),
        played: json['games_played'].toString(),
        won: json['wins'].toString(),
        draw: json['draws'].toString(),
        lost: json['losses'].toString(),
        goalsScored: json['goals_scored'].toString(),
        goalsAgainst: json['goals_against'].toString(),
        goalDifference: json['goal_difference'].toString(),
        points: json['points'].toString(),
        imageUrl: 'http://192.168.100.15/slpl_app/${json['image']}',
      );
}
