// ignore_for_file: use_build_context_synchronously

import 'dart:convert';
import 'package:flutter/material.dart';
import 'package:http/http.dart' as http;

import 'addresult.dart';
import 'config.dart';
import 'createleague.dart';
import 'createteams.dart';
import 'importfixtures.dart';
import 'main.dart';

void main() {
  runApp(AdminPage());
}

class AdminPage extends StatelessWidget {
  @override
  Widget build(BuildContext context) {
    return MaterialApp(
      title: 'Flutter Football App',
      theme: ThemeData(
        primarySwatch: Colors.blue,
      ),
      home: LeagueAdmin(),
    );
  }
}

class LeagueAdmin extends StatefulWidget {
  @override
  _LeagueAdminState createState() => _LeagueAdminState();
}

class _LeagueAdminState extends State<LeagueAdmin> {
  String imageUrl = '';
  String title = '';
  String description = '';
  String? errorMessage;

  @override
  void initState() {
    super.initState();
    fetchData();
  }

  Future<void> fetchData() async {
    try {
      final response =
          await http.get(Uri.parse('${AppConfig.baseUrl}/getleague.php'));
      if (response.statusCode == 200) {
        final data = json.decode(response.body);
        setState(() {
          title = data['title'];
          const SizedBox(
            height: 50,
          );
          imageUrl = data['image_url']
              .replaceAll('\\/', '/'); // Remove escape characters

          description = data['description'];
        });
      } else {
        throw Exception('Failed to load data');
      }
    } catch (e) {
      errorMessage = 'No data found';
    }
  }

  @override
  Widget build(BuildContext context) {
    return Scaffold(
      appBar: AppBar(
          backgroundColor: Color.fromARGB(255, 16, 99, 60),
          foregroundColor: const Color.fromARGB(255, 252, 252, 252),
          leading: IconButton(
            icon: Icon(Icons.arrow_back),
            onPressed: () {
              // Navigate back when the back button is pressed
              Navigator.push(
                context,
                MaterialPageRoute(builder: (context) => const SlplApp()),
              );
            },
          ),
          title: Container(
            decoration: BoxDecoration(
              borderRadius: BorderRadius.circular(8),
            ),
            child: const Text(
              "Current League",
              style: TextStyle(
                color: Color.fromARGB(255, 246, 247, 248),
                fontSize: 18,
                fontWeight: FontWeight.bold,
              ),
            ),
          ),
          actions: [
            PopupMenuButton(
              itemBuilder: (context) {
                return [
                  const PopupMenuItem<int>(
                    value: 0,
                    child: Text(
                      "View All League",
                      style: TextStyle(fontWeight: FontWeight.bold),
                    ),
                  ),
                  const PopupMenuItem<int>(
                    value: 1,
                    child: Text(
                      "View Teams",
                      style: TextStyle(fontWeight: FontWeight.bold),
                    ),
                  ),
                  const PopupMenuItem<int>(
                    value: 2,
                    child: Text(
                      "Add Teams To A league",
                      style: TextStyle(fontWeight: FontWeight.bold),
                    ),
                  ),
                  const PopupMenuItem<int>(
                    value: 3,
                    child: Text(
                      "Add Fixtures",
                      style: TextStyle(fontWeight: FontWeight.bold),
                    ),
                  ),
                  const PopupMenuItem<int>(
                    value: 4,
                    child: Text(
                      "Set Result",
                      style: TextStyle(fontWeight: FontWeight.bold),
                    ),
                  ),
                ];
              },
              onSelected: (value) {
                // Handle menu item selection  ViewLeagues
                if (value == 0) {
                  Navigator.push(
                    context,
                    MaterialPageRoute(
                        builder: (context) => const ViewLeagues()),
                  );
                }
                // Handle menu item selection

                if (value == 1) {
                  Navigator.push(
                    context,
                    MaterialPageRoute(builder: (context) => const ViewTeams()),
                  );
                }
                if (value == 2) {
                  Navigator.push(
                    context,
                    MaterialPageRoute(
                        builder: (context) => TeamSelectionScreen()),
                  );
                }

                if (value == 3) {
                  Navigator.push(
                    context,
                    MaterialPageRoute(
                        builder: (context) => const ImportFixtures()),
                  );
                }
                if (value == 4) {
                  Navigator.push(
                    context,
                    MaterialPageRoute(
                        builder: (context) => const ViewResults()),
                  );
                }

                // Handle menu item selection
              },
            ),
          ]),
      body: ListView(
        children: [
          GestureDetector(
            child: Container(
              alignment: Alignment.center,
              child: Column(
                mainAxisAlignment: MainAxisAlignment.center,
                children: [
                  SizedBox(height: 20),
                  Text(
                    title,
                    style: const TextStyle(
                        fontSize: 20,
                        color: Color.fromARGB(255, 61, 61, 66),
                        fontWeight: FontWeight.bold),
                  ),
                  SizedBox(height: 20),
                  Image.network(
                    imageUrl,
                    width: 200,
                    height: 200,
                    errorBuilder: (context, error, stackTrace) {
                      return const Placeholder(
                        fallbackWidth: 40,
                        fallbackHeight: 40,
                        color: Colors.grey, // Placeholder color
                      );
                    },
                  ),
                  SizedBox(height: 40),
                  const SizedBox(
                    height: 5,
                  ),
                  Container(
                    width: 400,
                    padding: const EdgeInsets.all(5),
                    child: Center(
                      child: Text(
                        description,
                        textAlign: TextAlign.center,
                        style: const TextStyle(
                          fontSize: 15,
                          color: Color.fromARGB(255, 136, 133, 133),
                          fontWeight: FontWeight.bold,
                        ),
                      ),
                    ),
                  ),
                ],
              ),
            ),
          ),
        ],
      ),
    );
  }
}

class Team {
  final String id;
  final String name;

  bool isSelected;

  Team({required this.id, required this.name, this.isSelected = false});

  factory Team.fromJson(Map<String, dynamic> json) {
    return Team(
      id: json['id'],
      name: json['team_name'],
    );
  }
}

class TeamSelectionScreen extends StatefulWidget {
  @override
  _TeamSelectionScreenState createState() => _TeamSelectionScreenState();
}

class _TeamSelectionScreenState extends State<TeamSelectionScreen> {
  List<Team> teams = [];

  String? selectedGroup;
  List<String> leagueGroups = [];

  // List of leagues (replace with your actual list)

  @override
  void initState() {
    super.initState();
    fetchTeams();
    fetchleagues();
  }

  Future<void> fetchleagues() async {
    final response =
        await http.get(Uri.parse('${AppConfig.baseUrl}fetch_leagues.php'));

    if (response.statusCode == 200) {
      final dynamic jsonData = json.decode(response.body);

      if (jsonData is Map<String, dynamic> &&
          jsonData.containsKey('groups') &&
          jsonData['groups'] is List) {
        setState(() {
          leagueGroups = List<String>.from(jsonData['groups']);
          selectedGroup = leagueGroups.isNotEmpty ? leagueGroups[0] : null;
        });
      }
    } else {
      print('Error fetching League groups: ${response.statusCode}');
    }
  }

  Future<void> fetchTeams() async {
    final response = await http
        .get(Uri.parse('${AppConfig.baseUrl}select_teams_for_league.php'));
    if (response.statusCode == 200) {
      final List<dynamic> responseData = jsonDecode(response.body);
      setState(() {
        teams = responseData.map((json) => Team.fromJson(json)).toList();
      });
    } else {
      // Handle error
      print('Failed to load teams');
    }
  }

  Future<void> submitSelectedTeams() async {
    List<Team> selectedTeams = teams.where((team) => team.isSelected).toList();
    // Now you have a list of selected teams, you can post their data
    // For example:
    for (var team in selectedTeams) {
      // Post team data to your backend
      // Replace this with your actual HTTP post logic
      const String url = '${AppConfig.baseUrl}add_team_to_league.php';

      try {
        final response = await http.post(
          Uri.parse(url),
          body: {
            'selectedTeams': team.name,
            'team_id': team.id,
            'league_name': selectedGroup,
          },
        );

        if (response.statusCode == 200) {
          Navigator.pushReplacement(
            context,
            MaterialPageRoute(
              builder: (context) => const ViewTeams(),
            ),
          );
        } else {
          // Handle other response statuses
          print('Error: ${response.statusCode}');
        }
      } catch (e) {
        // Handle network or other errors
        print('Error: $e');
      }
    }
  }

  @override
  Widget build(BuildContext context) {
    return Scaffold(
      appBar: AppBar(
        iconTheme: const IconThemeData(color: Colors.white),
        backgroundColor: Color.fromARGB(255, 16, 99, 60),
        title: const Text(
          'Select Teams for League',
          style: TextStyle(color: Colors.white),
        ),
      ),
      body: Column(
        children: [
          // Dropdown for selecting league

          DropdownButtonFormField<String>(
            value: selectedGroup,
            onChanged: (String? newValue) {
              setState(() {
                selectedGroup = newValue!;
              });
            },
            items: leagueGroups.map((String value) {
              return DropdownMenuItem<String>(
                value: value,
                child: Text(value),
              );
            }).toList(),
            decoration: const InputDecoration(
              labelText: 'Selected League',
              hintStyle: TextStyle(fontWeight: FontWeight.bold),
            ),
          ),

          Expanded(
            child: ListView.builder(
              itemCount: teams.length,
              itemBuilder: (context, index) {
                final team = teams[index];
                return ListTile(
                  title: Row(
                    children: [
                      Checkbox(
                        value: team.isSelected,
                        onChanged: (value) {
                          setState(() {
                            team.isSelected = value ?? false;
                          });
                        },
                      ),
                      Text(
                        team.name,
                        style: const TextStyle(
                          color: Color.fromARGB(255, 6, 51, 18),
                          fontWeight: FontWeight.bold,
                        ),
                      )
                    ],
                  ),
                  onTap: () {
                    setState(() {
                      team.isSelected = !team.isSelected;
                    });
                  },
                );
              },
            ),
          ),
        ],
      ),
      floatingActionButton: FloatingActionButton(
        onPressed: () {
          submitSelectedTeams();
        },
        child: Icon(Icons.check),
        backgroundColor: Colors.green,
        foregroundColor: Colors.white,
      ),
    );
  }
}
